﻿#include "precompiled.h"
#include "common.h"
#include "RTRenderer.h"

#include "RTCore.h"
#include "Camera.h"
#include "Lens.h"
#include "Aperture.h"
#include "Film.h"
#include "Scene.h"
#include "Entity.h"
#include "Transform.h"
#include "RenderTarget.h"
#include "DepthStencil.h"
#include "Texture1D.h"

#include "MeshRenderer.h"
#include "Mesh.h"

#include "VertexShader.h"
#include "PixelShader.h"
#include "GeometryShader.h"

//#define ENABLE_MOTION_BLUR_TEST

// TODO: Refactor usage of mipmapping function calls

using namespace DirectX;

const UINT maxAnisotropy = 16;

const float xOffset = 5;
const float zOffset = 20;
const float frontRowYOffset = -1;
const float backRowYOffset = 20;
const float backRowZOffset = -500;
const float cubeScale = 2;
const float backCubeScale = 6;

// Daylight = 1000 lux
// Twilight = 5 lux
// Sky color = 0, 0.05, 0.2
//#define NIGHT_COLOR
const float backgroundClearColor[] = { 0.0f, 50.0f, 200.0f, 1.0f }; // HDR values
const float backgroundClearColor2[] = { 0.0f, 0.25f, 1.0f, 1.0f }; // HDR values

const float planeDepth = 1;

const int bokehDownsampleFactor = 2;

namespace RTCam {

RTRenderer::RTRenderer() :
	m_deviceReady(false),
	m_cameraActive(false),
	m_cameraWarningPrinted(false),
	m_renderingLongExposure(false),
	m_debugRenderMode(0),
	m_numBokehPoints(0),
	m_currentLongExposureDuration(0),
	m_cameraMotionReset(true),
	m_drawUI(false),
	m_d2dUISize(),
	// Rendertargets
	m_intermediateRT(new RenderTarget("IntermediateRT")),
	m_intermediateDS(new DepthStencil("IntermediateDS")),
	m_depthBokehRadiusRT(new RenderTarget("LinearDepthAndBokehSizeRT")),
	m_velocityMapRT(new RenderTarget("VelocityMapRT")),
	m_bokehFgBlurRT(new RenderTarget("ForegroundBokehRT")),
	m_bokehBgBlurRT(new RenderTarget("BackgroundBokehRT")),
	m_bokehBlurTempRT(new RenderTarget("TempBokehRT")),
	m_postprocessRT1(new RenderTarget("PostprocessRT1")),
	m_postprocessRT2(new RenderTarget("PostprocessRT2")),
	m_longExposureRT(new RenderTarget("LongExposureRT")),
	// Shaders
	m_initialVShader(new VertexShader()),
	m_initialPShader(new PixelShader()),
	m_depthBokehRadiusPShader(new PixelShader()),
	m_dofVShader(new VertexShader()),
	m_dofGShader(new GeometryShader()),
	m_dofPShader(new PixelShader()),
	m_dofMergePShader(new PixelShader()),
	m_aberrationFirstPassPShader(new PixelShader()),
	m_aberrationSecondPassPShader(new PixelShader()),
	m_motionBlurPShader(new PixelShader()),
	m_longExposurePShader(new PixelShader()),
	m_fullscreenVShader(new VertexShader()),
	m_fullscreenDebugVShader(new VertexShader()),
	m_oldDoFPShader(new PixelShader()),
	m_tonemapPShader(new PixelShader()),
	m_tonemapGrainPShader(new PixelShader()),
	m_boxBlurHorizPShader(new PixelShader()),
	m_boxBlurVertPShader(new PixelShader()),
	// Test shaders
	m_testVShader(new VertexShader()),
	m_testPShader(new PixelShader()),
	m_CoCPShader(new PixelShader()),
	m_depthPShader(new PixelShader()),
	m_velocityPShader(new PixelShader()),
	m_copyPShader(new PixelShader()),
	// Input layouts
	m_posNormalColorLayout(nullptr),
	m_posTexLayout(nullptr),
	// Samplers and blend states
	m_borderedLinearSampler(nullptr),
	m_blendAdditive(nullptr),
	m_blendNonpremultiplied(nullptr),
	// Vertex and index buffers
	m_fullscreenQuadVertBuffer(nullptr),
	m_fullscreenQuadIndexBuffer(nullptr),
	m_bokehVertBuffer(nullptr),
	// Constant buffers
	m_modelCBuffer(nullptr),
	m_cameraCBuffer(nullptr)
{
}


RTRenderer::~RTRenderer(void)
{
}

void RTRenderer::CreateDeviceResources()
{
	Direct3DBase::CreateDeviceResources();

	m_userAnnotations->BeginEvent(L"RTRenderer: CreateDeviceResources");

	CreateShadersAndBindInputLayouts();
	CreateConstantBuffers();
	CreateFullscreenQuadBuffers();
	CreateSamplerAndBlendStates();

	m_deviceReady = true;

	m_userAnnotations->EndEvent();
}


void RTRenderer::CreateConstantBuffers()
{
	// Create the constant buffers
	CD3D11_BUFFER_DESC modelCBufferDesc(sizeof(ModelCBuffer), D3D11_BIND_CONSTANT_BUFFER);
	ThrowIfFailed(
		m_d3dDevice->CreateBuffer(
		&modelCBufferDesc,
		nullptr,
		&m_modelCBuffer
		)
		);
	SetDebugObjectName(m_modelCBuffer.Get(), "Model Constant Buffer");
	CD3D11_BUFFER_DESC cameraCBufferDesc(sizeof(CameraCBuffer), D3D11_BIND_CONSTANT_BUFFER);
	ThrowIfFailed(
		m_d3dDevice->CreateBuffer(
		&cameraCBufferDesc,
		nullptr,
		&m_cameraCBuffer
		)
		);
	SetDebugObjectName(m_cameraCBuffer.Get(), "Camera Constant Buffer");
}


void RTRenderer::CreateSamplerAndBlendStates()
{
	// Create custom sampler and blend states
	auto textureAddressMode = D3D11_TEXTURE_ADDRESS_BORDER;
	CD3D11_SAMPLER_DESC linearSamplerDesc(
		//D3D11_FILTER_MIN_MAG_MIP_LINEAR,
		D3D11_FILTER_ANISOTROPIC,
		textureAddressMode,
		textureAddressMode,
		textureAddressMode,
		0.0f,							// LOD bias
		maxAnisotropy,
		D3D11_COMPARISON_NEVER,
		Colors::Black,
		0,								// Min LOD
		D3D11_FLOAT32_MAX				// Max LOD
		);
	ThrowIfFailed(
		m_d3dDevice->CreateSamplerState(
		&linearSamplerDesc,
		&m_borderedLinearSampler
		)
		);
	SetDebugObjectName(m_borderedLinearSampler.Get(), "Linear Sampler with Black Border");

	// Additive blend
	D3D11_BLEND_DESC blendDesc = {0};
	blendDesc.IndependentBlendEnable = false;
	
	blendDesc.RenderTarget[0].BlendEnable		= true;
	blendDesc.RenderTarget[0].SrcBlend			= D3D11_BLEND_SRC_ALPHA;
	blendDesc.RenderTarget[0].SrcBlendAlpha	= D3D11_BLEND_ONE;
	blendDesc.RenderTarget[0].DestBlend		= D3D11_BLEND_ONE;
	blendDesc.RenderTarget[0].DestBlendAlpha	= D3D11_BLEND_ONE;
	blendDesc.RenderTarget[0].BlendOp			= D3D11_BLEND_OP_ADD;
	blendDesc.RenderTarget[0].BlendOpAlpha		= D3D11_BLEND_OP_ADD;
	blendDesc.RenderTarget[0].RenderTargetWriteMask = D3D11_COLOR_WRITE_ENABLE_ALL;
	
	ThrowIfFailed(m_d3dDevice->CreateBlendState(&blendDesc, &m_blendAdditive));
	SetDebugObjectName(m_blendAdditive.Get(), "Additive Blend State");

	// Nonpremultiplied alpha blend
	blendDesc.RenderTarget[0].BlendEnable		= true;
	blendDesc.RenderTarget[0].SrcBlend			= D3D11_BLEND_SRC_ALPHA;
	blendDesc.RenderTarget[0].SrcBlendAlpha	= D3D11_BLEND_ONE;
	blendDesc.RenderTarget[0].DestBlend		= D3D11_BLEND_INV_SRC_ALPHA;
	blendDesc.RenderTarget[0].DestBlendAlpha	= D3D11_BLEND_INV_SRC_ALPHA;
	blendDesc.RenderTarget[0].BlendOp			= D3D11_BLEND_OP_ADD;
	blendDesc.RenderTarget[0].BlendOpAlpha		= D3D11_BLEND_OP_ADD;
	blendDesc.RenderTarget[0].RenderTargetWriteMask = D3D11_COLOR_WRITE_ENABLE_ALL;

	ThrowIfFailed(m_d3dDevice->CreateBlendState(&blendDesc, &m_blendNonpremultiplied));
	SetDebugObjectName(m_blendNonpremultiplied.Get(), "Nonpremultiplied Alpha Blend State");
}


void RTRenderer::CreateShadersAndBindInputLayouts() 
{
	auto initialVSBlob = m_initialVShader->Load("InitialVShader", m_d3dDevice.Get());
	m_initialPShader->Load("InitialPShader", m_d3dDevice.Get());

	m_depthBokehRadiusPShader->Load("DepthBokehRadiusPShader", m_d3dDevice.Get());

	m_dofVShader->Load("DoFVShader", m_d3dDevice.Get());
	m_dofGShader->Load("DoFGShader", m_d3dDevice.Get());
	m_dofPShader->Load("DoFPShader", m_d3dDevice.Get());
	m_dofMergePShader->Load("DoFMergePShader", m_d3dDevice.Get());
	m_aberrationFirstPassPShader->Load("AberrationFirstPassPShader", m_d3dDevice.Get());
	m_aberrationSecondPassPShader->Load("AberrationSecondPassPShader", m_d3dDevice.Get());
	m_motionBlurPShader->Load("MotionBlurPShader", m_d3dDevice.Get());
	m_longExposurePShader->Load("LongExposurePShader", m_d3dDevice.Get());

	m_fullscreenVShader->Load("FullscreenVShader", m_d3dDevice.Get());
	auto fullscreenDebugVSBlob = m_fullscreenDebugVShader->Load("FullscreenDebugVShader", m_d3dDevice.Get());
	m_oldDoFPShader->Load("OldDoFPShader", m_d3dDevice.Get());

	m_tonemapPShader->Load("TonemapPShader", m_d3dDevice.Get());
	m_tonemapGrainPShader->Load("TonemapGrainPShader", m_d3dDevice.Get());

	m_boxBlurHorizPShader->Load("BoxBlurHorizPShader", m_d3dDevice.Get());
	m_boxBlurVertPShader->Load("BoxBlurVertPShader", m_d3dDevice.Get());
	
	m_testVShader->Load("TestVShader", m_d3dDevice.Get());
	m_testPShader->Load("TestPShader", m_d3dDevice.Get());
	m_CoCPShader->Load("CoCPShader", m_d3dDevice.Get());
	m_depthPShader->Load("DepthPShader", m_d3dDevice.Get());
	m_velocityPShader->Load("VelocityPShader", m_d3dDevice.Get());
	m_copyPShader->Load("CopyPShader", m_d3dDevice.Get());

	// Define and create input layouts
	// TODO: Make shaders share input layouts using shader header files
	ThrowIfFailed(
		m_d3dDevice->CreateInputLayout(
		VertexPositionNormalColor::InputElements,
		VertexPositionNormalColor::InputElementCount,
		initialVSBlob->GetBufferPointer(),
		initialVSBlob->GetBufferSize(),
		&m_posNormalColorLayout
		)
		);
	SetDebugObjectName(m_posNormalColorLayout.Get(), "PosNormalColor Layout");

	ThrowIfFailed(
		m_d3dDevice->CreateInputLayout(
		VertexPositionTexture::InputElements,
		VertexPositionTexture::InputElementCount,
		fullscreenDebugVSBlob->GetBufferPointer(),
		fullscreenDebugVSBlob->GetBufferSize(),
		&m_posTexLayout
		)
		);
	SetDebugObjectName(m_posTexLayout.Get(), "PosTex Layout");
}


void RTRenderer::ReleaseResources()
{
	Direct3DBase::ReleaseResources();

	m_intermediateRT->ReleaseResources();
	m_intermediateDS->ReleaseResources();
	m_bokehFgBlurRT->ReleaseResources();
	m_bokehBgBlurRT->ReleaseResources();
	m_bokehBlurTempRT->ReleaseResources();

	m_bokehVertBuffer = nullptr;
	m_numBokehPoints = 0;
}


void RTRenderer::CreateWindowSizeDependentResources()
{
	m_userAnnotations->BeginEvent(L"RTRenderer: Create Window Size Dependent Resources");

	Direct3DBase::CreateWindowSizeDependentResources();
	auto device = m_d3dDevice.Get();

	// Create the depth stencil and render target for the first pass.
	m_intermediateRT->Init(device, m_renderSize.x, m_renderSize.y, RenderTarget::rtTexFormatHDR);
	m_intermediateDS->Init(device, m_renderSize.x, m_renderSize.y);

	// Create the render target for the linear depth/bokeh size prepass.
	m_depthBokehRadiusRT->Init(device, m_renderSize.x, m_renderSize.y, DXGI_FORMAT_R16G16_FLOAT);

	// Create the render target for the velocity map.
	m_velocityMapRT->Init(device, m_renderSize.x, m_renderSize.y, DXGI_FORMAT_R16G16_FLOAT);

	// Create the render targets to store the outputs of postprocess passes.
	m_postprocessRT1->Init(device, m_renderSize.x, m_renderSize.y, RenderTarget::rtTexFormatHDRHalf);
	m_postprocessRT2->Init(device, m_renderSize.x, m_renderSize.y, RenderTarget::rtTexFormatHDRHalf);

	// Create the render target to store the long exposure image.
	m_longExposureRT->Init(device, m_renderSize.x, m_renderSize.y, RenderTarget::rtTexFormatHDR);

	CreateBokehResources();

#ifdef ENABLE_DIRECT2D
	m_d2dUISize = m_d2dBackbufferRT->GetSize();
#endif

	m_userAnnotations->EndEvent();
}

void RTRenderer::UpdateForWindowSizeChange()
{
	Direct3DBase::UpdateForWindowSizeChange();
	ResetLongExposureResources();
}

void RTRenderer::RenderUpdate( const shared_ptr<Scene>& scene, float curTime, float delta )
{
	auto camera = scene->m_activeCam.lock();
	m_currentCamera = camera;
	
	m_cameraActive = camera != nullptr;
	if(m_cameraActive) {
		// Update the camera's constant buffer data
		XMMATRIX viewMatrix;
		XMMATRIX projMatrix;
		XMMATRIX viewProjMatrix;
		camera->UpdateConstantBuffer(m_renderSize, curTime, delta, viewMatrix, projMatrix, viewProjMatrix);

		if(m_renderingLongExposure) {
			m_currentLongExposureDuration += delta;
			
			// Update the blend factor
			camera->m_cameraCBufferData.LongExposureBlendFactor = delta / m_currentLongExposureDuration;
		}

		// Update the world-view-projection matrices of objects in the scene
		for(auto& e : scene->m_entities) {
			UpdateWorldViewProjectionMatrices(e, viewProjMatrix, m_cameraMotionReset);
		}

		m_cameraMotionReset = false;

#ifdef ENABLE_MOTION_BLUR_TEST
		// MOTION BLUR TEST CODE
		for(auto& e : scene->m_entities) {
			if(e->GetCamera() != nullptr) {
				continue;
			}

			auto transform = e->GetTransform();
			auto position = transform->GetLocalPosition();

			position.x = sin(curTime * 2) * 2;
			position.y = cos(curTime * 2) * 2;

			transform->SetLocalPosition(position);
		}
		// END TEST
#endif

		// Make a warning print the next time there's no active camera.
		m_cameraWarningPrinted = false;
	} else {
		if(!m_cameraWarningPrinted) {
			DebugPrint("No camera active.\n");
			m_cameraWarningPrinted = true;
		}
		return;
	}
}


void RTRenderer::UpdateWorldViewProjectionMatrices(const shared_ptr<Entity>& entity, CXMMATRIX viewProjection, bool resetCameraMotion)
{
	auto transform = entity->GetTransform();
	ASSERT(transform != nullptr);

	transform->UpdateWorldViewProjectionMatrix(viewProjection, resetCameraMotion);

	for(auto& child : entity->GetChildren()) {
		ASSERT(child != nullptr);
		UpdateWorldViewProjectionMatrices(child, viewProjection, resetCameraMotion);
	}
}


void RTRenderer::Render(const shared_ptr<Scene>& scene)
{
	if (!m_deviceReady) {
		return;
	}

	if(!m_cameraActive) {
		m_deviceContext->ClearRenderTargetView(m_finalRenderTargetView.Get(), Colors::Magenta);
		return;
	}

	m_userAnnotations->BeginEvent(L"Render");

	auto camera = m_currentCamera.lock();
	ASSERT(camera == scene->m_activeCam.lock());

	// Set the camera constant buffer data
	m_deviceContext->UpdateSubresource(m_cameraCBuffer.Get(), 0, nullptr, &camera->m_cameraCBufferData, 0, 0);

	// Set sampler states
	ID3D11SamplerState* samplers[] = {
		m_states->PointClamp(),
		m_states->LinearClamp(),
		m_states->AnisotropicClamp(),
		m_borderedLinearSampler.Get()
	};
	m_deviceContext->VSSetSamplers(0, ARRAYSIZE(samplers), samplers);
	m_deviceContext->GSSetSamplers(0, ARRAYSIZE(samplers), samplers);
	m_deviceContext->PSSetSamplers(0, ARRAYSIZE(samplers), samplers);

	InitialPass(scene);
	DepthAndBokehPrepass();

	switch(m_debugRenderMode) {
	case 1:
		// Tonemapped but otherwise unprocessed
		TonemapPass(m_intermediateRT->m_shaderView.Get(), m_finalRenderTargetView.Get());
		break;
	case 2:
		// Depth values
		RenderDepth();
		break;
	case 3:
		// CoC radii
		RenderCoCRadius();
		break;
	case 4:
		// Old Reverse-mapped DoF
		OldDoFPass();
		break;
	case 5:
		// Foreground DoF blur only
		BokehPass(m_intermediateRT->m_shaderView.Get());
		BokehBlurPass();
		TonemapPass(m_bokehFgBlurRT->m_shaderView.Get(), m_finalRenderTargetView.Get());
		break;
	case 6:
		// Background DoF blur only
		BokehPass(m_intermediateRT->m_shaderView.Get());
		BokehBlurPass();
		TonemapPass(m_bokehBgBlurRT->m_shaderView.Get(), m_finalRenderTargetView.Get());
		break;
	case 7:
		// Velocity only
		RenderVelocity();
		break;
	case 8:
		// View the last recorded long exposure image
		TonemapPass(m_longExposureRT->m_shaderView.Get(), m_finalRenderTargetView.Get());
		break;
	case 0:
	default:
		MotionBlurPass(m_intermediateRT->m_shaderView.Get(), m_postprocessRT1->m_renderTargetView.Get());
		BokehPass(m_postprocessRT1->m_shaderView.Get());
		BokehBlurPass();
		DoFMergePass(m_postprocessRT2->m_renderTargetView.Get());
		AberrationFirstPass(m_postprocessRT2->m_shaderView.Get(), m_postprocessRT1->m_renderTargetView.Get());
		AberrationSecondPass(m_postprocessRT1->m_shaderView.Get(), m_postprocessRT2->m_renderTargetView.Get());
		
		if(m_renderingLongExposure) {
			LongExposurePass(m_postprocessRT2->m_shaderView.Get(), m_longExposureRT->m_renderTargetView.Get());
			TonemapGrainPass(m_longExposureRT->m_shaderView.Get(), m_finalRenderTargetView.Get());
		} else {
			// Display the result immediately
			TonemapGrainPass(m_postprocessRT2->m_shaderView.Get(), m_finalRenderTargetView.Get());
		}
		break;
	}

	UIPass();

	m_userAnnotations->EndEvent();
}

void RTRenderer::CreateFullscreenQuadBuffers()
{
	// TODO: Use a triangle that covers entire screen

	// Define vertex data
	VertexPositionTexture planeVertices[] =
	{
		VertexPositionTexture(XMFLOAT3(-1, -1, planeDepth), XMFLOAT2(0, 1)), // 0 lower left
		VertexPositionTexture(XMFLOAT3(-1,  1, planeDepth), XMFLOAT2(0, 0)), // 1 upper left
		VertexPositionTexture(XMFLOAT3( 1,  1, planeDepth), XMFLOAT2(1, 0)), // 2 upper right
		VertexPositionTexture(XMFLOAT3( 1, -1, planeDepth), XMFLOAT2(1, 1)), // 3 lower right
	};

	// Define polygon indices
	uint16_t planeIndices[] = 
	{
		0, 1, 2,
		0, 2, 3
	};

	// Create vertex buffer
	D3D11_SUBRESOURCE_DATA vertexBufferData = {0};
	vertexBufferData.pSysMem = planeVertices;
	vertexBufferData.SysMemPitch = 0;
	vertexBufferData.SysMemSlicePitch = 0;
	CD3D11_BUFFER_DESC vertexBufferDesc(sizeof(planeVertices), D3D11_BIND_VERTEX_BUFFER);
	ThrowIfFailed(
		m_d3dDevice->CreateBuffer(
		&vertexBufferDesc,
		&vertexBufferData,
		&m_fullscreenQuadVertBuffer
		)
		);
	SetDebugObjectName(m_fullscreenQuadVertBuffer.Get(), "Fullscreen Quad Vertex Buffer");

	// Create the index buffer
	D3D11_SUBRESOURCE_DATA indexBufferData = {0};
	indexBufferData.pSysMem = planeIndices;
	indexBufferData.SysMemPitch = 0;
	indexBufferData.SysMemSlicePitch = 0;
	CD3D11_BUFFER_DESC indexBufferDesc(sizeof(planeIndices), D3D11_BIND_INDEX_BUFFER);
	ThrowIfFailed(
		m_d3dDevice->CreateBuffer(
		&indexBufferDesc,
		&indexBufferData,
		&m_fullscreenQuadIndexBuffer
		)
		);
	SetDebugObjectName(m_fullscreenQuadIndexBuffer.Get(), "Fullscreen Quad Index Buffer");
}

void RTRenderer::DrawScene(const shared_ptr<Scene>& scene)
{
	// TODO: Batch render calls by mesh used.
	for(auto& e : scene->m_entities) {
		DrawEntity(e);
	}
}

void RTRenderer::DrawEntity(const shared_ptr<Entity>& entity)
{
	ASSERT_MSG(entity != nullptr, "A null entity was passed to DrawEntity");
	auto renderer = entity->GetRenderer();

	if(renderer != nullptr) {
		shared_ptr<Mesh> mesh = renderer->GetMesh();
		if(mesh != nullptr) {

			auto topology = D3D11_PRIMITIVE_TOPOLOGY_TRIANGLELIST;
			auto inputLayout = m_posNormalColorLayout.Get();

			// Prepare the constant buffers
			m_deviceContext->UpdateSubresource(m_modelCBuffer.Get(), 0, nullptr, &entity->m_modelCBufferData, 0, 0);
			ID3D11Buffer* buffers[] = {
				m_modelCBuffer.Get()
			};

			// Prepare the vertex and index buffers
			ID3D11Buffer* vertexBuffers[] = {
				mesh->m_vertexBuffer.Get()
			};
			UINT stride = sizeof(VertexPositionNormalColor);
			UINT offset = 0;
			auto indexBuffer = mesh->m_indexBuffer.Get();

			// -------------------- Input-Assembler --------------------
			m_deviceContext->IASetPrimitiveTopology(topology);
			m_deviceContext->IASetInputLayout(inputLayout);
			m_deviceContext->IASetVertexBuffers(0, ARRAYSIZE(vertexBuffers), vertexBuffers, &stride, &offset);
			m_deviceContext->IASetIndexBuffer(indexBuffer, DXGI_FORMAT_R16_UINT, 0);

			// -------------------- Shaders --------------------
			m_deviceContext->VSSetConstantBuffers(1, ARRAYSIZE(buffers), buffers);
			m_deviceContext->PSSetConstantBuffers(1, ARRAYSIZE(buffers), buffers);

			// Draw
			m_deviceContext->DrawIndexed(mesh->m_indexCount, 0, 0);
		} else if(!renderer->m_meshWarningPrinted) {
			DebugPrint("WARNING: Entity %s has a mesh renderer with no mesh attached.\n", entity->m_name.c_str());
			renderer->m_meshWarningPrinted = true;
		}
	}

	// Draw each child
	for(auto& child : entity->GetChildren()) {
		ASSERT_MSG(child != nullptr, "An entity had a null child.");
		DrawEntity(child);
	}
}

void RTRenderer::CreateBokehResources()
{
	// Use a lower resolution buffer to speed up rendering.
	int bokehRTWidth = m_renderSize.x / bokehDownsampleFactor;
	int bokehRTHeight = m_renderSize.y / bokehDownsampleFactor;

	// Make sure the resolution is a multiple of 2
	if(bokehRTWidth % 2 == 1)
		++bokehRTWidth;
	if(bokehRTHeight % 2 == 1)
		++bokehRTHeight;

	// Create the render targets for the bokeh pass.
	m_bokehFgBlurRT->Init(m_d3dDevice.Get(), bokehRTWidth, bokehRTHeight, RenderTarget::rtTexFormatHDR);
	m_bokehBgBlurRT->Init(m_d3dDevice.Get(), bokehRTWidth, bokehRTHeight, RenderTarget::rtTexFormatHDR);
	m_bokehBlurTempRT->Init(m_d3dDevice.Get(), bokehRTWidth, bokehRTHeight, RenderTarget::rtTexFormatHDR);

	// One vertex will be sent to the geometry shader for each group of 2x2 pixels
	int numBokehX = bokehRTWidth / 2;
	int numBokehY = bokehRTHeight / 2;
	m_numBokehPoints = numBokehX * numBokehY;

	// The distance between neighboring pixels in clip space.
	// (Clip space goes from -1 to 1)
	float posDeltaX = 2.0f / bokehRTWidth;
	float posDeltaY = 2.0f / bokehRTHeight;

	// The distance between neighboring pixels in texture coordinate space.
	// (Clip space goes from 0 to 1)
	float texDeltaX = 1.0f / bokehRTWidth;
	float texDeltaY = 1.0f / bokehRTHeight;

	// At the top-left corner, x = -1, y = 1, u = 0, v = 0.
	// Vertex centers are further offset by one pixel size (so that they're at the center of each 2x2 group).
	float posInitX = -1 + posDeltaX;
	float posInitY = 1 - posDeltaY;
	float texInitX = texDeltaX;
	float texInitY = texDeltaY;

	// Define vertex data (one per group of 2x2 pixels)
	unique_ptr<VertexPositionTexture[]> bokehVertices(new VertexPositionTexture[m_numBokehPoints]);
	for(int y = 0; y < numBokehY; ++y) {
		for(int x = 0; x < numBokehX; ++x) {
			float posX = posInitX + (x * posDeltaX * 2);
			float posY = posInitY - (y * posDeltaY * 2);
			float texU = texInitX + (x * texDeltaX * 2);
			float texV = texInitY + (y * texDeltaY * 2);

			int index = y * numBokehX + x;
			bokehVertices[index] = VertexPositionTexture(XMFLOAT3(posX, posY, 0), XMFLOAT2(texU, texV));
		}
	}
	
	// Create vertex buffer
	D3D11_SUBRESOURCE_DATA vertexBufferData = {0};
	vertexBufferData.pSysMem = bokehVertices.get();
	vertexBufferData.SysMemPitch = 0;
	vertexBufferData.SysMemSlicePitch = 0;
	CD3D11_BUFFER_DESC vertexBufferDesc(sizeof(VertexPositionTexture) * m_numBokehPoints, D3D11_BIND_VERTEX_BUFFER);
	ThrowIfFailed(
		m_d3dDevice->CreateBuffer(
		&vertexBufferDesc,
		&vertexBufferData,
		&m_bokehVertBuffer
		)
		);
	SetDebugObjectName(m_bokehVertBuffer.Get(), "Bokeh Vertex Buffer");

	DebugPrint(L"Num quads in bokeh pass: %i+\n", m_numBokehPoints);
}

//--------------------------------------------------------------------------------------

void RTRenderer::InitialPass(const shared_ptr<Scene>& scene) 
{
	m_userAnnotations->BeginEvent(L"Initial Pass");
	auto blending = m_states->NonPremultiplied();

	auto intermediateRT = m_intermediateRT->m_renderTargetView.Get();
	auto velocityRT = m_velocityMapRT->m_renderTargetView.Get();
	auto depthStencil = m_intermediateDS->m_depthStencilView.Get();
	auto vertexShader = m_initialVShader->m_vertexShader.Get();
	auto pixelShader = m_initialPShader->m_pixelShader.Get();

	ID3D11RenderTargetView* rendertargets[] = {
		intermediateRT,
		velocityRT
	};
	ID3D11Buffer* buffers[] = {
		m_cameraCBuffer.Get()
	};

	// Clear the render targets and depth stencil
#ifdef NIGHT_COLOR
	m_deviceContext->ClearRenderTargetView(intermediateRT, backgroundClearColor2);
#else
	m_deviceContext->ClearRenderTargetView(intermediateRT, backgroundClearColor);
#endif
	
	m_deviceContext->ClearRenderTargetView(velocityRT, Colors::Transparent);
	m_deviceContext->ClearDepthStencilView(depthStencil, D3D11_CLEAR_DEPTH | D3D11_CLEAR_STENCIL, 1.0f, 0);

	// -------------------- Vertex Shader --------------------
	m_deviceContext->VSSetShader(vertexShader, nullptr, 0);
	m_deviceContext->VSSetConstantBuffers(0, ARRAYSIZE(buffers), buffers);

	// -------------------- Pixel Shader --------------------
	m_deviceContext->PSSetShader(pixelShader, nullptr, 0);
	m_deviceContext->PSSetConstantBuffers(0, ARRAYSIZE(buffers), buffers);

	// -------------------- Output Merger --------------------
	m_deviceContext->OMSetRenderTargets(ARRAYSIZE(rendertargets), rendertargets, depthStencil);
	m_deviceContext->OMSetBlendState(blending, nullptr, 0xFFFFFFFF);
	m_deviceContext->OMSetDepthStencilState(m_states->DepthDefault(), 0);

	// -------------------- Draw call --------------------
	DrawScene(scene);

	// -------------------- Cleanup --------------------

	// Unbind rendertargets
	for(auto& rt : rendertargets) {
		rt = nullptr;
	}
	m_deviceContext->OMSetRenderTargets(ARRAYSIZE(rendertargets), rendertargets, nullptr);

	// Unbind shaders
	m_deviceContext->VSSetShader(nullptr, nullptr, 0);
	m_deviceContext->PSSetShader(nullptr, nullptr, 0);

	m_userAnnotations->EndEvent();
}

void RTRenderer::DepthAndBokehPrepass()
{
	m_userAnnotations->BeginEvent(L"Depth and Bokeh Prepass");

	ID3D11ShaderResourceView* views[] = { m_intermediateDS->m_shaderView.Get() };
	ID3D11Buffer* buffers[] = { m_cameraCBuffer.Get() };
	ID3D11RenderTargetView* rtViews[] = { m_depthBokehRadiusRT->m_renderTargetView.Get() };

	// Prepare the rendertarget.
	m_deviceContext->ClearRenderTargetView(m_depthBokehRadiusRT->m_renderTargetView.Get(), Colors::Transparent);

	FullscreenParams params;
	params.pixelShader = m_depthBokehRadiusPShader->m_pixelShader.Get();
	params.views = views;
	params.numViews = ARRAYSIZE(views);
	params.buffers = buffers;
	params.numBuffers = ARRAYSIZE(buffers);
	params.blendState = m_states->Opaque();
	params.rtViews = rtViews;
	params.numRTViews = ARRAYSIZE(rtViews);

	FullscreenPass(params);

	m_userAnnotations->EndEvent();
}

//--------------------------------------------------------------------------------------

void RTRenderer::BokehPass(_In_ ID3D11ShaderResourceView* source)
{
	m_userAnnotations->BeginEvent(L"Bokeh Pass");

	auto camera = m_currentCamera.lock();
	ASSERT(camera != nullptr);

	auto positiveSATex = camera->m_lens->m_positiveSATex;
	auto zeroSATex = camera->m_lens->m_zeroSATex;
	auto negativeSATex = camera->m_lens->m_negativeSATex;
	ASSERT(positiveSATex != nullptr && zeroSATex != nullptr && negativeSATex != nullptr);

	auto bokehTex = camera->m_aperture->m_bokehTex;
	ASSERT(bokehTex != nullptr);

	auto topology = D3D11_PRIMITIVE_TOPOLOGY_POINTLIST;
	auto layout = m_posTexLayout.Get();
	auto blending = m_blendAdditive.Get();

	ID3D11ShaderResourceView* views[] = {
		source,
		m_depthBokehRadiusRT->m_shaderView.Get(),
		positiveSATex->m_shaderView.Get(),
		zeroSATex->m_shaderView.Get(),
		negativeSATex->m_shaderView.Get(),
		bokehTex->m_shaderView.Get()
	};
	ID3D11Buffer* buffers[] = {
		m_cameraCBuffer.Get()
	};

	ASSERT(m_bokehFgBlurRT->GetWidth() == m_bokehBgBlurRT->GetWidth());
	ASSERT(m_bokehFgBlurRT->GetHeight() == m_bokehBgBlurRT->GetHeight());

	auto fgRenderTarget = m_bokehFgBlurRT->m_renderTargetView.Get();
	auto bgRenderTarget = m_bokehBgBlurRT->m_renderTargetView.Get();
	auto depthStencil = nullptr;

	ID3D11RenderTargetView* rendertargets[] = {
		fgRenderTarget,
		bgRenderTarget
	};
	m_deviceContext->ClearRenderTargetView(fgRenderTarget, Colors::Transparent);
	m_deviceContext->ClearRenderTargetView(bgRenderTarget, Colors::Transparent);

	// Prepare the vertex buffer
	ID3D11Buffer* vertexBuffers[] = { m_bokehVertBuffer.Get() };
	UINT stride = sizeof(VertexPositionTexture);
	UINT offset = 0;

	// Set viewport
	CD3D11_VIEWPORT viewportNew(
		0.0f,
		0.0f,
		static_cast<float>(m_bokehFgBlurRT->GetWidth()),
		static_cast<float>(m_bokehFgBlurRT->GetHeight())
		);
	m_deviceContext->RSSetViewports(1, &viewportNew);

	// Generate mip maps.
	m_deviceContext->GenerateMips(m_intermediateRT->m_shaderView.Get());

	// -------------------- Input-Assembler --------------------

	// Set the input layout and topology
	m_deviceContext->IASetPrimitiveTopology(topology);
	m_deviceContext->IASetInputLayout(layout);
	m_deviceContext->IASetVertexBuffers(0, ARRAYSIZE(vertexBuffers), vertexBuffers, &stride, &offset);

	// -------------------- Vertex Shader --------------------
	m_deviceContext->VSSetShader(m_dofVShader->m_vertexShader.Get(), nullptr, 0);
	m_deviceContext->VSSetConstantBuffers(0, ARRAYSIZE(buffers), buffers);

	// -------------------- Geometry Shader --------------------
	m_deviceContext->GSSetShader(m_dofGShader->m_geometryShader.Get(), nullptr, 0);
	m_deviceContext->GSSetShaderResources(0, ARRAYSIZE(views), views);
	m_deviceContext->GSSetConstantBuffers(0, ARRAYSIZE(buffers), buffers);

	// -------------------- Pixel Shader --------------------
	m_deviceContext->PSSetShader(m_dofPShader->m_pixelShader.Get(), nullptr, 0);
	m_deviceContext->PSSetShaderResources(0, ARRAYSIZE(views), views);
	m_deviceContext->PSSetConstantBuffers(0, ARRAYSIZE(buffers), buffers);

	// -------------------- Output Merger --------------------
	m_deviceContext->OMSetRenderTargets(ARRAYSIZE(rendertargets), rendertargets, depthStencil);
	m_deviceContext->OMSetBlendState(blending, nullptr, 0xFFFFFFFF);
	m_deviceContext->OMSetDepthStencilState(m_states->DepthNone(), 0);

	// -------------------- Draw call --------------------
	m_deviceContext->Draw(m_numBokehPoints, 0);

	// -------------------- Cleanup --------------------

	// Unbind shader resource views
	for(auto& view : views) {
		view = nullptr;
	}
	m_deviceContext->GSSetShaderResources(0, ARRAYSIZE(views), views);
	m_deviceContext->PSSetShaderResources(0, ARRAYSIZE(views), views);

	// Unbind rendertargets
	for(auto& rt : rendertargets) {
		rt = nullptr;
	}
	m_deviceContext->OMSetRenderTargets(ARRAYSIZE(rendertargets), rendertargets, nullptr);

	// Unbind shaders
	m_deviceContext->VSSetShader(nullptr, nullptr, 0);
	m_deviceContext->GSSetShader(nullptr, nullptr, 0);
	m_deviceContext->PSSetShader(nullptr, nullptr, 0);

	// Reset viewport
	CD3D11_VIEWPORT viewport(
		0.0f,
		0.0f,
		static_cast<float>(m_renderSize.x),
		static_cast<float>(m_renderSize.y)
		);
	m_deviceContext->RSSetViewports(1, &viewport);

	m_userAnnotations->EndEvent();
}

void RTRenderer::BokehBlurPass()
{
	// Set viewport
	CD3D11_VIEWPORT viewportNew(
		0.0f,
		0.0f,
		static_cast<float>(m_bokehFgBlurRT->GetWidth()),
		static_cast<float>(m_bokehFgBlurRT->GetHeight())
		);
	m_deviceContext->RSSetViewports(1, &viewportNew);

	BoxBlurPass(m_bokehFgBlurRT, m_bokehBlurTempRT);
	BoxBlurPass(m_bokehBgBlurRT, m_bokehBlurTempRT);

	// Reset viewport
	CD3D11_VIEWPORT viewport(
		0.0f,
		0.0f,
		static_cast<float>(m_renderSize.x),
		static_cast<float>(m_renderSize.y)
		);
	m_deviceContext->RSSetViewports(1, &viewport);
}


//--------------------------------------------------------------------------------------

void RTRenderer::FullscreenPass(const FullscreenParams& params)
{
	// -------------------- Input-Assembler and Vertex Shader --------------------
	m_deviceContext->IASetPrimitiveTopology(D3D11_PRIMITIVE_TOPOLOGY_TRIANGLELIST);

#ifdef _DEBUG
	// In debug mode, explicitly pass in positions to enable shader debugging
	UINT stride = sizeof(VertexPositionTexture);
	UINT offset = 0;
	ID3D11Buffer* vertexBuffers = m_fullscreenQuadVertBuffer.Get();
	m_deviceContext->IASetVertexBuffers(0, 1, &vertexBuffers, &stride, &offset);
	m_deviceContext->IASetIndexBuffer(m_fullscreenQuadIndexBuffer.Get(), DXGI_FORMAT_R16_UINT, 0);
	
	m_deviceContext->IASetInputLayout(m_posTexLayout.Get());
	m_deviceContext->VSSetShader(m_fullscreenDebugVShader->m_vertexShader.Get(), nullptr, 0);
#else
	m_deviceContext->IASetInputLayout(nullptr);
	m_deviceContext->VSSetShader(m_fullscreenVShader->m_vertexShader.Get(), nullptr, 0);
#endif

	// -------------------- Pixel Shader --------------------
	m_deviceContext->PSSetShader(params.pixelShader, nullptr, 0);
	m_deviceContext->PSSetShaderResources(0, params.numViews, params.views);
	if(params.numBuffers != 0) {
		m_deviceContext->PSSetConstantBuffers(0, params.numBuffers, params.buffers);
	}

	// -------------------- Output Merger --------------------
	m_deviceContext->OMSetRenderTargets(params.numRTViews, params.rtViews, nullptr);
	m_deviceContext->OMSetBlendState(params.blendState, nullptr, 0xFFFFFFFF);
	m_deviceContext->OMSetDepthStencilState(m_states->DepthNone(), 0);

	// -------------------- Draw call --------------------
#ifdef _DEBUG
	m_deviceContext->DrawIndexed(6, 0, 0);
#else
	m_deviceContext->Draw(6, 0);
#endif

	// -------------------- Cleanup --------------------

	// Unbind shader resource views
	unique_ptr<ID3D11ShaderResourceView*[]> nullShaderViews(new ID3D11ShaderResourceView*[params.numViews]);
	for(size_t i = 0; i < params.numViews; ++i) {
		nullShaderViews[i] = nullptr;
	}
	m_deviceContext->PSSetShaderResources(0, params.numViews, nullShaderViews.get());

	// Unbind rendertargets
	unique_ptr<ID3D11RenderTargetView*[]> nullRTViews(new ID3D11RenderTargetView*[params.numRTViews]);
	for(size_t i = 0; i < params.numRTViews; ++i) {
		nullRTViews[i] = nullptr;
	}
	m_deviceContext->OMSetRenderTargets(params.numRTViews, nullRTViews.get(), nullptr);

	// Unbind shaders
	m_deviceContext->VSSetShader(nullptr, nullptr, 0);
	m_deviceContext->PSSetShader(nullptr, nullptr, 0);
}


void RTRenderer::TonemapPass(_In_ ID3D11ShaderResourceView* shaderView, _In_ ID3D11RenderTargetView* output)
{
	m_userAnnotations->BeginEvent(L"Tonemap Pass");

	ID3D11ShaderResourceView* views[] = { shaderView };
	ID3D11Buffer* buffers[] = { m_cameraCBuffer.Get() };
	ID3D11RenderTargetView* rtViews[] = { output };

	// Prepare the rendertarget.
	m_deviceContext->ClearRenderTargetView(output, Colors::Transparent);

	FullscreenParams params;
	params.pixelShader = m_tonemapPShader->m_pixelShader.Get();
	params.views = views;
	params.numViews = ARRAYSIZE(views);
	params.buffers = buffers;
	params.numBuffers = ARRAYSIZE(buffers);
	params.blendState = m_states->Opaque();
	params.rtViews = rtViews;
	params.numRTViews = ARRAYSIZE(rtViews);

	FullscreenPass(params);

	m_userAnnotations->EndEvent();
}

void RTRenderer::TonemapGrainPass( _In_ ID3D11ShaderResourceView* shaderView, _In_ ID3D11RenderTargetView* output )
{
	m_userAnnotations->BeginEvent(L"Tonemap and Grain Pass");

	ID3D11ShaderResourceView* views[] = { shaderView };
	ID3D11Buffer* buffers[] = { m_cameraCBuffer.Get() };
	ID3D11RenderTargetView* rtViews[] = { output };

	// Prepare the rendertarget.
	m_deviceContext->ClearRenderTargetView(output, Colors::Transparent);

	FullscreenParams params;
	params.pixelShader = m_tonemapGrainPShader->m_pixelShader.Get();
	params.views = views;
	params.numViews = ARRAYSIZE(views);
	params.buffers = buffers;
	params.numBuffers = ARRAYSIZE(buffers);
	params.blendState = m_states->Opaque();
	params.rtViews = rtViews;
	params.numRTViews = ARRAYSIZE(rtViews);

	FullscreenPass(params);

	m_userAnnotations->EndEvent();
}


void RTRenderer::BoxBlurPass(const unique_ptr<RenderTarget>& sourceRT, const unique_ptr<RenderTarget>& intermediateRT)
{
	ASSERT(sourceRT->GetWidth() == intermediateRT->GetWidth());
	ASSERT(sourceRT->GetHeight() == m_bokehBgBlurRT->GetHeight());

	//----- Horizontal pass -----
	m_userAnnotations->BeginEvent(L"Box Blur Horizontal Pass");

	ID3D11ShaderResourceView* views[] = { sourceRT->m_shaderView.Get() };
	ID3D11Buffer* buffers[] = { m_cameraCBuffer.Get() };
	ID3D11RenderTargetView* rtViews[] = { intermediateRT->m_renderTargetView.Get() };

	// Prepare the rendertarget.
	m_deviceContext->ClearRenderTargetView(intermediateRT->m_renderTargetView.Get(), Colors::Transparent);

	FullscreenParams params;
	params.pixelShader = m_boxBlurHorizPShader->m_pixelShader.Get();
	params.views = views;
	params.numViews = ARRAYSIZE(views);
	params.buffers = buffers;
	params.numBuffers = ARRAYSIZE(buffers);
	params.blendState = m_states->Opaque();
	params.rtViews = rtViews;
	params.numRTViews = ARRAYSIZE(rtViews);

	FullscreenPass(params);

	m_userAnnotations->EndEvent();

	//----- Vertical pass -----
	m_userAnnotations->BeginEvent(L"Box Blur Vertical Pass");

	views[0] = intermediateRT->m_shaderView.Get();
	rtViews[0] = sourceRT->m_renderTargetView.Get();

	// Prepare the rendertarget.
	m_deviceContext->ClearRenderTargetView(sourceRT->m_renderTargetView.Get(), Colors::Transparent);

	params.pixelShader = m_boxBlurVertPShader->m_pixelShader.Get();
	
	FullscreenPass(params);

	m_userAnnotations->EndEvent();
}


void RTRenderer::CopyPass(_In_ ID3D11ShaderResourceView* shaderView, _In_ ID3D11RenderTargetView* output)
{
	m_userAnnotations->BeginEvent(L"Copy Pass");

	ID3D11ShaderResourceView* views[] = { shaderView };
	ID3D11RenderTargetView* rtViews[] = { output };

	// Prepare the rendertarget.
	m_deviceContext->ClearRenderTargetView(output, Colors::Transparent);

	FullscreenParams params;
	params.pixelShader = m_copyPShader->m_pixelShader.Get();
	params.views = views;
	params.numViews = ARRAYSIZE(views);
	params.buffers = nullptr;
	params.numBuffers = 0;
	params.blendState = m_states->Opaque();
	params.rtViews = rtViews;
	params.numRTViews = ARRAYSIZE(rtViews);

	FullscreenPass(params);

	m_userAnnotations->EndEvent();
}

void RTRenderer::RenderDepth()
{
	m_userAnnotations->BeginEvent(L"Render Depth");

	ID3D11ShaderResourceView* views[] = { m_depthBokehRadiusRT->m_shaderView.Get() };
	ID3D11Buffer* buffers[] = { m_cameraCBuffer.Get() };
	ID3D11RenderTargetView* rtViews[] = { m_finalRenderTargetView.Get() };

	// Prepare the rendertarget.
	m_deviceContext->ClearRenderTargetView(m_finalRenderTargetView.Get(), Colors::Transparent);

	FullscreenParams params;
	params.pixelShader = m_depthPShader->m_pixelShader.Get();
	params.views = views;
	params.numViews = ARRAYSIZE(views);
	params.buffers = buffers;
	params.numBuffers = ARRAYSIZE(buffers);
	params.blendState = m_states->Opaque();
	params.rtViews = rtViews;
	params.numRTViews = ARRAYSIZE(rtViews);

	FullscreenPass(params);

	m_userAnnotations->EndEvent();
}

void RTRenderer::RenderVelocity()
{
	m_userAnnotations->BeginEvent(L"Render Velocity");

	ID3D11ShaderResourceView* views[] = { m_velocityMapRT->m_shaderView.Get() };
	ID3D11Buffer* buffers[] = { m_cameraCBuffer.Get() };
	ID3D11RenderTargetView* rtViews[] = { m_finalRenderTargetView.Get() };

	// Prepare the rendertarget.
	m_deviceContext->ClearRenderTargetView(m_finalRenderTargetView.Get(), Colors::Transparent);

	FullscreenParams params;
	params.pixelShader = m_velocityPShader->m_pixelShader.Get();
	params.views = views;
	params.numViews = ARRAYSIZE(views);
	params.buffers = buffers;
	params.numBuffers = ARRAYSIZE(buffers);
	params.blendState = m_states->Opaque();
	params.rtViews = rtViews;
	params.numRTViews = ARRAYSIZE(rtViews);

	FullscreenPass(params);

	m_userAnnotations->EndEvent();
}

void RTRenderer::RenderCoCRadius()
{
	m_userAnnotations->BeginEvent(L"Render CoC Radius");

	ID3D11ShaderResourceView* views[] = { m_depthBokehRadiusRT->m_shaderView.Get() };
	ID3D11RenderTargetView* rtViews[] = { m_finalRenderTargetView.Get() };

	// Prepare the rendertarget.
	m_deviceContext->ClearRenderTargetView(m_finalRenderTargetView.Get(), Colors::Transparent);

	FullscreenParams params;
	params.pixelShader = m_CoCPShader->m_pixelShader.Get();
	params.views = views;
	params.numViews = ARRAYSIZE(views);
	params.buffers = nullptr;
	params.numBuffers = 0;
	params.blendState = m_states->Opaque();
	params.rtViews = rtViews;
	params.numRTViews = ARRAYSIZE(rtViews);

	FullscreenPass(params);

	m_userAnnotations->EndEvent();
}

void RTRenderer::OldDoFPass()
{
	m_userAnnotations->BeginEvent(L"Old DoF Blur Pass");

	ID3D11ShaderResourceView* views[] = {
		m_intermediateRT->m_shaderView.Get(),
		m_intermediateDS->m_shaderView.Get()
	};
	ID3D11Buffer* buffers[] = { m_cameraCBuffer.Get() };
	ID3D11RenderTargetView* rtViews[] = { m_finalRenderTargetView.Get() };

	// Prepare the rendertarget.
	m_deviceContext->ClearRenderTargetView(m_finalRenderTargetView.Get(), Colors::Transparent);

	// Generate mip maps.
	m_deviceContext->GenerateMips(m_intermediateRT->m_shaderView.Get());

	FullscreenParams params;
	params.pixelShader = m_oldDoFPShader->m_pixelShader.Get();
	params.views = views;
	params.numViews = ARRAYSIZE(views);
	params.buffers = buffers;
	params.numBuffers = ARRAYSIZE(buffers);
	params.blendState = m_states->Opaque();
	params.rtViews = rtViews;
	params.numRTViews = ARRAYSIZE(rtViews);

	FullscreenPass(params);

	m_userAnnotations->EndEvent();
}

void RTRenderer::DoFMergePass(_In_ ID3D11RenderTargetView* destination)
{
	m_userAnnotations->BeginEvent(L"DoF Merge Pass");

	ID3D11ShaderResourceView* views[] = {
		m_intermediateRT->m_shaderView.Get(),
		m_bokehFgBlurRT->m_shaderView.Get(),
		m_bokehBgBlurRT->m_shaderView.Get(),
		m_depthBokehRadiusRT->m_shaderView.Get()
	};
	ID3D11Buffer* buffers[] = { m_cameraCBuffer.Get() };
	ID3D11RenderTargetView* rtViews[] = { destination };

	// Prepare the rendertarget.
	m_deviceContext->ClearRenderTargetView(destination, Colors::Transparent);

	FullscreenParams params;
	params.pixelShader = m_dofMergePShader->m_pixelShader.Get();
	params.views = views;
	params.numViews = ARRAYSIZE(views);
	params.buffers = buffers;
	params.numBuffers = ARRAYSIZE(buffers);
	params.blendState = m_states->Opaque();
	params.rtViews = rtViews;
	params.numRTViews = ARRAYSIZE(rtViews);

	FullscreenPass(params);

	m_userAnnotations->EndEvent();
}

void RTRenderer::MotionBlurPass(_In_ ID3D11ShaderResourceView* source, _In_ ID3D11RenderTargetView* destination)
{
	m_userAnnotations->BeginEvent(L"Motion Blur Pass");

	ID3D11ShaderResourceView* views[] = {
		source,
		m_velocityMapRT->m_shaderView.Get()
	};
	ID3D11Buffer* buffers[] = { m_cameraCBuffer.Get() };
	ID3D11RenderTargetView* rtViews[] = { destination };

	// Prepare the rendertarget.
	m_deviceContext->ClearRenderTargetView(destination, Colors::Transparent);

	FullscreenParams params;
	params.pixelShader = m_motionBlurPShader->m_pixelShader.Get();
	params.views = views;
	params.numViews = ARRAYSIZE(views);
	params.buffers = buffers;
	params.numBuffers = ARRAYSIZE(buffers);
	params.blendState = m_states->NonPremultiplied();
	params.rtViews = rtViews;
	params.numRTViews = ARRAYSIZE(rtViews);

	FullscreenPass(params);

	m_userAnnotations->EndEvent();
}

void RTRenderer::AberrationFirstPass(_In_ ID3D11ShaderResourceView* source, _In_ ID3D11RenderTargetView* destination)
{
	m_userAnnotations->BeginEvent(L"Aberration First Pass");
	
	auto camera = m_currentCamera.lock();
	ASSERT(camera != nullptr);

	auto tangentialTex = camera->m_lens->m_tangentialTex;
	auto distortionTex = camera->m_lens->m_distortionTex;
	ASSERT(tangentialTex != nullptr && distortionTex != nullptr);

	ID3D11ShaderResourceView* views[] = {
		source,
		tangentialTex->m_shaderView.Get(),
		distortionTex->m_shaderView.Get()
	};
	ID3D11Buffer* buffers[] = { m_cameraCBuffer.Get() };
	ID3D11RenderTargetView* rtViews[] = { destination };

	// Prepare the rendertarget.
	m_deviceContext->ClearRenderTargetView(destination, Colors::Transparent);

	FullscreenParams params;
	params.pixelShader = m_aberrationFirstPassPShader->m_pixelShader.Get();
	params.views = views;
	params.numViews = ARRAYSIZE(views);
	params.buffers = buffers;
	params.numBuffers = ARRAYSIZE(buffers);
	params.blendState = m_states->Opaque();
	params.rtViews = rtViews;
	params.numRTViews = ARRAYSIZE(rtViews);

	FullscreenPass(params);

	m_userAnnotations->EndEvent();
}

void RTRenderer::AberrationSecondPass( _In_ ID3D11ShaderResourceView* source, _In_ ID3D11RenderTargetView* destination )
{
	m_userAnnotations->BeginEvent(L"Aberration Second Pass");

	auto camera = m_currentCamera.lock();
	ASSERT(camera != nullptr);

	auto sagittalTex = camera->m_lens->m_sagittalTex;
	auto vignettingTex = camera->m_lens->m_vignettingTexs[0].vignettingTex; // TODO: Find the closest texture
	ASSERT(sagittalTex != nullptr && vignettingTex != nullptr);

	ID3D11ShaderResourceView* views[] = {
		source,
		sagittalTex->m_shaderView.Get(),
		vignettingTex->m_shaderView.Get()
	};
	ID3D11Buffer* buffers[] = { m_cameraCBuffer.Get() };
	ID3D11RenderTargetView* rtViews[] = { destination };

	// Prepare the rendertarget.
	m_deviceContext->ClearRenderTargetView(destination, Colors::Transparent);

	FullscreenParams params;
	params.pixelShader = m_aberrationSecondPassPShader->m_pixelShader.Get();
	params.views = views;
	params.numViews = ARRAYSIZE(views);
	params.buffers = buffers;
	params.numBuffers = ARRAYSIZE(buffers);
	params.blendState = m_states->Opaque();
	params.rtViews = rtViews;
	params.numRTViews = ARRAYSIZE(rtViews);

	FullscreenPass(params);

	m_userAnnotations->EndEvent();
}

void RTRenderer::LongExposurePass( _In_ ID3D11ShaderResourceView* source, _In_ ID3D11RenderTargetView* destination )
{
	m_userAnnotations->BeginEvent(L"Long Exposure Pass");

	ID3D11ShaderResourceView* views[] = {
		source
	};
	ID3D11Buffer* buffers[] = { m_cameraCBuffer.Get() };
	ID3D11RenderTargetView* rtViews[] = { destination };

	
	FullscreenParams params;
	params.pixelShader = m_longExposurePShader->m_pixelShader.Get();
	params.views = views;
	params.numViews = ARRAYSIZE(views);
	params.buffers = buffers;
	params.numBuffers = ARRAYSIZE(buffers);
	params.blendState = m_blendNonpremultiplied.Get();
	params.rtViews = rtViews;
	params.numRTViews = ARRAYSIZE(rtViews);

	FullscreenPass(params);

	m_userAnnotations->EndEvent();
}

void RTRenderer::ToggleLongExposureRendering()
{
	m_renderingLongExposure = !m_renderingLongExposure;
	if(m_renderingLongExposure) {
		ResetLongExposureResources();
		DebugPrint("Now accumulating motion blur into a long exposure.\n");
	} else {
		DebugPrint("Long exposure turned off.\n");
	}
}

void RTRenderer::ResetLongExposureResources()
{
	// Clear the long exposure rendertarget
	m_deviceContext->ClearRenderTargetView(m_longExposureRT->m_renderTargetView.Get(), Colors::Transparent);

	// Reset the accumulated blur duration
	m_currentLongExposureDuration = 0;
}

void RTRenderer::UIPass()
{
	if(m_drawUI) {
#ifdef ENABLE_DIRECT2D
		m_d2dBackbufferRT->BeginDraw();
		m_d2dBackbufferRT->SetTransform(D2D1::Matrix3x2F::Identity());

		for(auto& queuedText : m_queuedText) {
			float x = queuedText.x;
			float y = queuedText.y;
			float width = queuedText.width;
			float height = queuedText.height;

			if(width == 0) {
				width = m_d2dUISize.width - x;
			}
			if(height == 0) {
				height = m_d2dUISize.height - y;
			}

			if(queuedText.anchor == UIRight) {
				x = m_d2dUISize.width - x - width;
			}

			DrawUIText(queuedText.text.c_str(), queuedText.text.size(), x, y, width, height);
		}

		ThrowIfFailed(m_d2dBackbufferRT->EndDraw());
#endif
	}
	
	m_queuedText.clear();
}

void RTRenderer::DrawUIText(const wchar_t* text, size_t length, float x, float y, float width, float height)
{
	const float shadowOffsetX = 1;
	const float shadowOffsetY = 2;

	// Do an outline effect by drawing the text twice
	float shadowX = x + shadowOffsetX;
	float shadowY = y + shadowOffsetY;

#ifdef ENABLE_DIRECT2D
	m_d2dBackbufferRT->DrawText(
		text,
		length,
		m_defaultTextFormat.Get(),
		D2D1::RectF(shadowX, shadowY, m_d2dUISize.width - shadowX, m_d2dUISize.height - shadowY),
		m_blackBrush.Get()
		);
	m_d2dBackbufferRT->DrawText(
		text,
		length,
		m_defaultTextFormat.Get(),
		D2D1::RectF(x, y, m_d2dUISize.width - x, m_d2dUISize.height - y),
		m_whiteBrush.Get()
		);
#endif
}

void RTRenderer::QueueUIText(const char* text, float x, float y, UIDirection anchor, float width, float height)
{
	UIText uitext;
	uitext.text = Utf8ToUtf16(text);
	uitext.x = x;
	uitext.y = y;
	uitext.anchor = anchor;
	uitext.width = width;
	uitext.height = height;

	m_queuedText.push_back(uitext);
}

} // end namespace